<?php

namespace App\Models\Tenant\Inventory;

use App\Models\Tenant\Branch;
use App\Models\Tenant\Inventory\Category;
use App\Models\Tenant\Inventory\Department;
use App\Models\Tenant\Inventory\Unit;
use App\Models\Tenant\Inventory\Stock;
use App\Models\Tenant\Inventory\UnitConversion;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

class Item extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'items';

    protected $fillable = [
        'item_code',
        'item_name',
        'category_id',
        'department_id',
        'item_type',
        'barcode',
        'location',
        'reorder_level',
        'opening_stock',
        'cost_price',
        'batch_number',
        'expiry_date',
        'status',
        'branch_id',
        'base_unit_id',
        'purchase_unit_id',
        'consume_unit_id',
        'is_deleted',
    ];

    protected $casts = [
        'expiry_date' => 'date',
        'status' => 'boolean',
        'is_deleted' => 'boolean',
        'opening_stock' => 'decimal:2',
        'cost_price' => 'decimal:2',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    /** Category of the item */
    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    /** Department associated with the item */
    public function department()
    {
        return $this->belongsTo(Department::class, 'department_id');
    }

    /** Type of the item */
    public function type()
    {
        return $this->belongsTo(ItemType::class, 'item_type');
    }

    /** Branch where the item belongs */
    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branch_id');
    }

    /** Base unit of measure */
    public function baseUnit()
    {
        return $this->belongsTo(Unit::class, 'base_unit_id');
    }

    /** Purchase unit of measure */
    public function purchaseUnit()
    {
        return $this->belongsTo(Unit::class, 'purchase_unit_id');
    }

    /** Consumption unit of measure */
    public function consumeUnit()
    {
        return $this->belongsTo(Unit::class, 'consume_unit_id');
    }

    /** All unit conversions related to this item */
    public function conversions()
    {
        return $this->hasMany(UnitConversion::class, 'item_id');
    }

    public function stocks()
    {
        return $this->hasMany(Stock::class, 'item_id');
    }

    public function getCost($batchNumber = null): float
    {
        $query = $this->stocks(); // use the relationship

        if ($batchNumber) {
            $query->where('id', $batchNumber);
        }

        $stock = $query->select('cost_price')->first();

        return $stock ? (float) $stock->cost_price : 0;
    }

    public function getAvailableStockAttribute()
    {
        return $this->stocks()->sum('quantity');
    }

    public static function adjustBatchQuantity(int $itemId, $batchNumber = null, float $quantity = 0, $branchId, $storeId = null)
    {
        // Normalize batch value (CI code treated 'null' string specially)
        $isBatchNull = ($batchNumber === null || $batchNumber === '' || $batchNumber === 'null');

        $query = DB::table('inventory_stock')
            ->where('item_id', $itemId);

        if ($isBatchNull) {
            // where batch_number is null
            $query->whereNull('batch_number');
        } else {
            $query->where('id', $batchNumber);
        }

        if (!empty($storeId)) {
            $query->where('store_id', $storeId);
        }

        $existing = $query->first();

        if ($existing) {
            // update existing
            return DB::table('inventory_stock')
                ->where('id', $existing->id)
                ->update([
                    'quantity' => $quantity,
                    'updated_at' => now(),
                ]) !== false;
        } else {
            // create new record
            return DB::table('inventory_stock')->insert([
                'item_id' => $itemId,
                'batch_number' => $isBatchNull ? null : $batchNumber,
                'store_id' => $storeId,
                'quantity' => $quantity,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }
}
