<?php

namespace App\Models\Tenant\Inventory;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\Tenant\Inventory\Item;
use App\Models\Tenant\Branch;
use Illuminate\Database\Eloquent\SoftDeletes;
use Exception;
use Illuminate\Support\Facades\DB;

class Stock extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'inventory_stock';
    protected $fillable = [
        'item_id',
        'store_id',
        'branch_id',
        'batch_number',
        'expiry_date',
        'serial_number',
        'quantity',
        'cost_price',
        'location',
    ];

    protected $casts = [
        'expiry_date' => 'date',
        'quantity' => 'decimal:2',
        'cost_price' => 'decimal:2',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function item()
    {
        return $this->belongsTo(Item::class);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * Example: if you have a separate store/warehouse table
     */
    public function store()
    {
        return $this->belongsTo(Store::class, 'store_id');
    }

   
    public static function adjustStoreStock($store_id, $item_id, $batch_id = null, $qty = 0, $item_cost = 0, $branch_id)
    {
        if (empty($store_id) || empty($item_id)) {
            throw new Exception("Store ID and Item ID are required.");
        }

        $qty = (float) $qty;

        return DB::transaction(function () use ($store_id, $item_id, $batch_id, $qty, $item_cost) {

            // 🔍 Try to find existing stock record
            $query = Stock::where('store_id', $store_id)
                ->where('item_id', $item_id);

            if (!empty($batch_id)) {
                $query->where('batch_number', $batch_id);
            } else {
                $query->whereNull('batch_number');
            }

            $existing = $query->first();

            if ($existing) {
                $newQty = $existing->quantity + $qty;

                if ($newQty < 0) {
                    throw new Exception("Insufficient stock {$qty} for item #{$item_id} in store #{$store_id}.");
                }

                Stock::where('id', $existing->id)
                    ->update([
                        'quantity' => $newQty,
                        'updated_at' => now(),
                    ]);

            } else {
                Stock::insert([
                    'store_id'     => $store_id,
                    'item_id'      => $item_id,
                    'batch_number' => $batch_id ?: null,
                    'quantity'     => $qty,
                    'cost_price'   => $item_cost,
                    'branch_id'    => $branch_id ?? 1,
                    'created_at'   => now(),
                    'updated_at'   => now(),
                ]);
            }

            return true;
        });
    }

    
}
